pro Mk_stdim_list, filename, list, coronagraph=coronagraph, $
                   exposure=exposure, filter=filter, polarizer=polarizer, $
                   pixels=pixels, dir_pattern=dir_pattern, $
                   interact=interact, upper_left=upper_left, save=save,  $
                   clevel=clevel, slevel=slevel, centre_pick=centre_pick

;+
; MK_STDIM_LIST
;	Form a "Standard" mask image from a list or more usually from
;	a set of parameters to be matched.
;
; Usage:
;	mk_stdim[, stdim, {list|<keys>}]
;
; Arguments:
;	filename string	input	The name to give to the image. If
;				unset, then use "stdim.fts"
;	list	string	input	An array with a list of filenames to
;				be used in forming the average image.
;
; Keywords: (all input)
;	coronagraph	int	- Which coronagraph (1, 2 or 3)
;	exposure	float	- 1-elements use exposures */10%,
;				2-elements use exposures in given
;				range, absent: use any exposure.
;	filter		string	- Specify the filter or FP setting
;				required (default "Clear")
;	polarizer	string	- Specify the polarizer setting to be
;				used (default clear)
;	pixels		int	- Specify the size of image to use
;                                 2-element array (default [1024,
;                                 1024]).
;	dir_pattern	string	- Specify a restricted set of
;                                 directories in which to find the
;                                 images (if used, this must be in a
;                                 form acceptable as an argument to
;                                 "find")
;	interact	??	- If set, then display each image
;                                 (reduced by a factor 2 if the larger
;                                 dimansion is > 512) and ask the user
;                                 whether to include it - tedious but
;                                                         sometimes
;                                                         needed.
;	upper_left	int	- Specify the location of the
;                                 upper-left corner of the image
;                                 (useful to distingush (say)
;                                 half-resolution-whole-field images
;                                 from full-resolution-quarter-field
;                                 images, 2-element array (default [20,1]).
;	save		??	- If set, then save the generated image
;				to disk
;	clevel		float	- Specify the minimum correlation with
;                                 the first image for the image to be
;                                 used. When this is specified the
;                                 first image is displayed as in
;                                 INTERACT and you are prompted whther
;                                 to use it; this continues until an
;                                 image is accepted. Thereafter all
;                                 images better correlated with the
;                                 initial image than the given value
;                                 are accepted - a typical level might
;                                                be in the range .98
;                                                to .99.
;	slevel		int	- Specify a pixel value to be
;                                 considered as saturated (default
;                                 16383). This is needed as some
;                                 images show "soft saturation" at a
;                                 level below 16383 counts.
;	centre_pick	??	- If set, then determine a Sun-centre
;                                 and radius, manually - else get a
;                                 centre from OCCLTR_CNTR
;
; Effects:
;	A "standard" average image is generated and added to the end
;	of the list of loaded images. Most of its header information
;	is derived from the first image of the set. N.B. The image is
;	NOT saved to disk, use one of the SAVE options in DISPLAY to
;	do this.
;
; WARNING:
;	It is the responsibility of the user to ensure that the image
;	set being used is homogeneous.
;
; History:
;	Original (modified from mk_stdim): 26/4/96, SJT
;	Substantially improved: 14/5/96; SJT
;-

@wload.com
@chandle.com

if (n_params() lt 2) then begin
    if (not keyword_set(coronagraph)) then  $
      message, "Must specify a list of files or a coronagraph ID"
    
    c = string(coronagraph, format = "('C',I1)")
    case (n_elements(exposure)) of
        0: exprange = [0., 10000.]
        1: exprange = exposure *[.9, 1.1]
        2: exprange = exposure
        Else: begin
            print, "*** W A R N I N G *** Too many elements in " + $
              "exposure using first 2"
            exprange = exposure(0:1)
        end
    endcase
    
    if (not keyword_set(filter)) then begin
        if (coronagraph eq 1) then filter = 'Fe XIV' $ ; Green line
        else filter = 'Clear'
    endif
    
    if (not keyword_set(polarizer)) then polarizer = 'Clear'
    
    if (not keyword_set(pixels)) then pixels = [1024, 1024]
    if (not keyword_set(upper_left)) then upper_left = [20, 1]
    
    image_dir = getenv("LASCO_IMAGES")
    if (keyword_set(dir_pattern)) then image_dir = image_dir + $
      !Delimiter + dir_pattern
    
    spawn, 'find '+image_dir+' -name img_hdr.txt -print', header_list
    
    list = ''
    total_images = 0
    for j = 0, n_elements(header_list)-1 do begin
        cdir = strmid(header_list(j), 0, rstrpos(header_list(j), $
                                                 !Delimiter)+1)
        
        
        cdlist = findfile(cdir+'*.fts') ; Need this as the
        cdlist = strmid(cdlist, strlen(cdir), 12) ; img_hdr.txt files
                                ; sometimes contain 
                                ; spurious files.
        
        
        openr, ilu, /get, header_list(j)
        
        fn = '' & date = '' & cg = '' & exptime = 0. 
        imsz = intarr(2) & upperleft = intarr(2) & ifilt = '' 
        ipol = '' & iprog = ''
        
        ihave = 0
        while (not eof(ilu)) do begin
            readf, ilu, fn, date, cg, exptime, imsz, upperleft, ifilt, $
              ipol, iprog, format =  $
              "(a12,4x,a20,3x,a2,f8.0,4i7,2x,2a7,a)"
            
            locs = where(fn eq cdlist, nmatch)
            if (nmatch ne 0 and c eq cg and $
                exprange(0) le exptime and exptime le exprange(1) and $
                imsz(0) eq pixels(0) and imsz(1) eq pixels(1) and $
                upperleft(0) eq upper_left(0) and upperleft(1) eq $
                upper_left(1) and $
                filter eq strtrim(ifilt, 2) and $
                polarizer eq strtrim(ipol, 2)) then begin
                
                ihave = ihave+1
                total_images = total_images+1
                list = [temporary(list), cdir+fn]
            endif
        endwhile
        
        free_lun, ilu
        if (ihave ne 0) then print, header_list(j), ihave
    endfor
    list = list(1:*)
    
    print, "Total number of images =", total_images
endif else if (N_elements(list) eq 1) then begin
    if (strpos(list, !Delimiter) eq -1) then  $
      list = getenv("LASCO_WORK") + !Delimiter + getenv("USER") + $
      !Delimiter + 'list' + !Delimiter + list
    openr, ilu, /get, list
    
    line = ''
    while (not eof(ilu)) do begin
        readf, ilu, line
        list = [list, line]
    endwhile
    
    free_lun, ilu
    list = list(1:*)
endif


if (keyword_set(interact) or keyword_set(clevel)) then begin
    window, xsize = 512, ysize = 512
    loadct, 0
endif

if (not keyword_set(slevel)) then slevel = 16383

used = 0l

for j = 0l, n_elements(list)-1 do begin
    
    img = 0                     ; memsave
    img = readfitsl(list(j), f_hdr, /silent)
    s = size(img)
    if (j eq 0) then begin
	stdim = fltarr(s(1), s(2))
        stexp = fltarr(s(1), s(2))
        hdr = f_hdr

    endif

    cg = strtrim(sxpar(f_hdr, 'DETECTOR'), 2)
    rp = strtrim(sxpar(f_hdr, 'READPORT'), 2)
    n_lebsum = sxpar(f_hdr, 'LEBXSUM')*sxpar(f_hdr, 'LEBYSUM') > 1
    
    dark = offset_bias(cg, rp)
    imin = min_nz(img, max = mx)

    print, j, list(j), s(1:2), imin, mx, $
      format = "(//i4,2x,a/2i5,2i8)"
    
    mask = 0                    ; Just a memory saving trick
    mask = mask_out(img, f_hdr)
    mask = rebin(temporary(mask), s(1), s(2), /samp)
    
    mask = mask and sigma_mask(img, 15, n_sigma = 4., /all, outbox = 3)
    mask = mask and (img lt slevel*n_lebsum)
    
    if (s(1) > s(2)) gt 512 then $
      imgd = bytscl(rebin(mask and float(img-dark), s(1)/2, s(2)/2), $
                    min = 0, top = !D.n_colors-1) $
    else imgd = bytscl(mask and float(img-dark), min = 0, top = $
                       !D.n_colors-1)
    
    if (keyword_set(interact)) then begin
        tv, imgd
        ans = ''
Ask:
        read, 'Use this image? (y/n) :_', ans
    endif else if (keyword_set(clevel)) then begin
        
        
        if (n_elements(ok1) eq 0) then begin
            tv, imgd
            ans = ''
Ask1:
            read, 'Use this image? (y/n) :_', ans
            if (strupcase(ans) eq 'Y') then begin
                ok1 = img
                mask1 = mask
            endif else if (strupcase(ans) ne 'N') then goto, ask1
        endif else begin
            cc = cc0(ok1, img,  mask = mask and mask1)
            print, cc
            if (cc gt clevel) then ans = 'Y' $
            else ans = 'N'
        endelse
    endif
    
    if strupcase(ans) eq 'Y' then begin
        
        im_exp = mask  and fxpar(f_hdr, 'EXPTIME')
        stexp = stexp + im_exp
        n_csum = sxpar(f_hdr, 'SUMROW')* sxpar(f_hdr, 'SUMCOL') > 1
        
        stdim = stdim + (mask * (img-dark)/float(c_psum))
        used = used+1
    endif else if (strupcase(ans) ne 'N') then goto, ask
    
endfor

locs = where(stexp eq 0., nzero)
if (nzero ne 0) then begin
    print, "**** ++++ Generated image still has " + $
	string(nzero,format="(I0)") + " holes."
    stexp(locs) = 1.
;;    stdim(locs) = !values.f_nan
endif

stdim = stdim/rebin(stexp, s(1), s(2), /samp)

mk_stdim_hdr, hdr, f_hdr, stdim, filename, used


image = float(temporary(stdim))
name = filename
head = hdr


;     Need to have a centre & radius for the occulting disk to allow
;	 masking options 

if (keyword_set(centre_pick)) then scan4limb, image, name, head, h_head, $
  index, 1, mini = min(image), maxi = max(image) $
else begin
    cntr = OCCLTR_CNTR(head)
    suncol = cntr(0)-(sxpar(head, 'R1COL')-20)
    sunrow = cntr(1)- sxpar(head, 'R1ROW')
    sxaddpar, head, 'CENTER_X', suncol
    sxaddpar, head, 'CENTER_Y', sunrow
endelse
  

sumrow = sxpar(head, 'SUMROW') > sxpar(head, 'LEBYSUM')
sumcol = sxpar(head, 'SUMCOL') > sxpar(head, 'LEBXSUM')
if (sumrow ne sumcol) then message, "E R R O R * * * Non-uniform " + $
  "stretching"

p1col = sxpar(head, 'P1COL')
p1row = sxpar(head, 'P1ROW')
center_x = sxpar(head, 'CENTER_X')
center_y = sxpar(head, 'CENTER_Y')
radius = sxpar(head, 'RADIUS')

radius = radius*sumrow
center_x = center_x*sumrow
center_y = center_y*sumrow

sxaddpar, head, 'CENTER_X', center_x
sxaddpar, head, 'CENTER_Y', center_y
sxaddpar, head, 'RADIUS', radius

chandle
if (n_elements(sel_img) gt 0) then sel_img = [sel_img, 0] $
else sel_img =  0

if (keyword_set(save)) then begin
            
    wdir = GETENV('WORK')+GETENV('USER')+!Delimiter+ 'fits'+!Delimiter
    IF chk_dir(wdir) EQ 0 THEN BEGIN
        cmd = 'mkdirhier '
        spawn, cmd + " " + wdir
    ENDIF
    wdir = wdir + !Delimiter
    writefits, wdir + name, image, head
endif

end
