function Mask_out, img, hdr

;+
; MASK_OUT
;	Determine missing telemetry blocks.
;
; Usage:
;	mask=mask_out(img,hdr)
;
; Return value:
;	mask	byte	A 2-d byte array with the same size as the
;			input image with zeroes in the telemetry holes
;			and ones elsewhere.
;
; Arguments:
;	img	any	input	The image to be analysed
;	hdr	string	input	The FITS header associated with the
;				image.
;
; Method:
;	The size of a telemetry block in the image is determined from
;	the SUMCOL or LEBYSUM, R?COL and the actual image size. The image is
;	rebinned to a 4-d array its minimum value is subtracted and
;	the resulting array is summed over the telemetry
;	blocks. Zeroes in the summed array are assumed to be missing
;	blocks. This is justified as (a) real data are +ve definite
;	and (b) the probability of getting the whole of a telemetry
;	block having the same value if there are real data present is
;	vanishingly small. This algorithm has the advantage of not
;	being dependent on whether the exposures have been normalized
;	or whether the camera bias has been subtracted.
;-

si = size(img)

mi = min(img, max = mx)

scale = si(1)*(sxpar(hdr, 'SUMCOL') > sxpar(hdr, 'LEBXSUM') > 1) / $
  float(sxpar(hdr, 'R2COL')-sxpar(hdr, 'R1COL'))
blksz = round(32*scale)

nbx = si(1)/blksz
nby = si(2)/blksz
mask = bytarr(nbx,nby)

im=reform(img, Blksz, nbx, Blksz, nby)-mi

imsums = total(total(im, 3), 1)

mask = (imsums ne 0.)
if (keyword_set(verbose)) then $
	print, mask, format="("+string(nbx)+"I1)"

mask = rebin(/samp, mask, nbx*Blksz, nby*Blksz)

return, mask
end
    

