function Mk_list,listfile, coronagraph=coronagraph, $
                  exposure=exposure, filter=filter, polarizer=polarizer, $
                  pixels=pixels, dir_pattern=dir_pattern, $
                  upper_left=upper_left


;+
; MK_LIST
;	Form a "Standard" mask image from a list or more usually from
;	a set of parameters to be matched.
;
; Usage:
;	list=mk_list( <keys> )
;
;
; Keywords: (all input)
;	coronagraph	int	- Which coronagraph (1, 2 or 3)
;	exposure	float	- 1-elements use exposures */10%,
;				2-elements use exposures in given
;				range, absent: use any exposure.
;	filter		string	- Specify the filter or FP setting
;				required (default "Clear")
;	polarizer	string	- Specify the polarizer setting to be
;				used (default clear)
;	pixels		int	- Specify the size of image to use
;                                 2-element array (default [1024,
;                                 1024]).
;	dir_pattern	string	- Specify a restricted set of
;                                 directories in which to find the
;                                 images (if used, this must be in a
;                                 form acceptable as an argument to
;                                 "find")
;	upper_left	int	- Specify the location of the
;                                 upper-left corner of the image
;                                 (useful to distingush (say)
;                                 half-resolution-whole-field images
;                                 from full-resolution-quarter-field
;                                 images, 2-element array (default [20,1]).
;
; Effects:
;	A list of files matching the specified properties is
;	generated.
;
; WARNING:
;	It is the responsibility of the user to ensure that the image
;	set being used is homogeneous.
;
; History:
;	Original (modified from mk_stdim_list): 15/5/96, SJT
;	Substantially improved: 14/5/96; SJT
;-

if (not keyword_set(coronagraph)) then  $
  message, "Must specify a coronagraph ID"

c = string(coronagraph, format = "('C',I1)")
case (n_elements(exposure)) of
    0: exprange = [0., 10000.]
    1: exprange = exposure *[.9, 1.1]
    2: exprange = exposure
    Else: begin
        print, "*** W A R N I N G *** Too many elements in " + $
          "exposure using first 2"
        exprange = exposure(0:1)
    end
endcase

if (not keyword_set(filter)) then begin
    if (coronagraph eq 1) then filter = 'Fe XIV' $ ; Green line
    else filter = 'Clear'
endif

if (not keyword_set(polarizer)) then polarizer = 'Clear'

if (not keyword_set(pixels)) then pixels = [1024, 1024]
if (not keyword_set(upper_left)) then upper_left = [20, 1]

if (not keyword_set(slevel)) then slevel = 16383

image_dir = getenv("LASCO_IMAGES")
if (keyword_set(dir_pattern)) then image_dir = image_dir + $
  !Delimiter + dir_pattern

spawn, 'find '+image_dir+' -name img_hdr.txt -print', header_list

list = ''
total_images = 0
for j = 0, n_elements(header_list)-1 do begin
    cdir = strmid(header_list(j), 0, rstrpos(header_list(j), $
                                             !Delimiter)+1)
    
    
    cdlist = findfile(cdir+'*.fts') ; Need this as the
    cdlist = strmid(cdlist, strlen(cdir), 12) ; img_hdr.txt files
                                ; sometimes contain 
                                ; spurious files.
    
    
    openr, ilu, /get, header_list(j)
    
    fn = '' & date = '' & cg = '' & exptime = 0. 
    imsz = intarr(2) & upperleft = intarr(2) & ifilt = '' 
    ipol = '' & iprog = ''
    
    ihave = 0
    while (not eof(ilu)) do begin
        readf, ilu, fn, date, cg, exptime, imsz, upperleft, ifilt, $
          ipol, iprog, format = $
          "(a12,4x,a20,3x,a2,f8.0,4i7,2x,2a7,a)"
        
        locs = where(fn eq cdlist, nmatch)
        if (nmatch ne 0 and c eq cg and $
            exprange(0) le exptime and exptime le exprange(1) and $
            imsz(0) eq pixels(0) and imsz(1) eq pixels(1) and $
            upperleft(0) eq upper_left(0) and upperleft(1) eq $
            upper_left(1) and $
            filter eq strtrim(ifilt, 2) and $
            polarizer eq strtrim(ipol, 2)) then begin
            
            ihave = ihave+1
            total_images = total_images+1
            list = [temporary(list), cdir+fn]
        endif
    endwhile
    
    free_lun, ilu
    print, header_list(j), ihave
endfor
list = list(1:*)

print, "Total number of images =", total_images

if (n_params() eq 0) then begin
    if (strpos(listfile, !Delimiter) eq -1) then $
      outfile = getenv('LASCO_WORK') + !Delimiter + getenv('USER') + $
      !Delimiter + 'list' + !Delimiter + listfile $
    else outfile = listfile
    
    on_ioerror, abort
    
    openw, ilu, /get, outfile
    printf, ilu, list, format = "(A)"
    free_lun, ilu
endif

Abort:

return, list

end
