pro Las_exp_norm, calib, store=store, all=all, $
                  use_in_mem=use_in_mem

;+
; LAS_EXP_NORM
;	Normalize images based on the integrated image intensities
;	
; Usage:
;	las_exp_norm, calib[, /store, /use_in_mem]
;
; Arguments:
;	calib	input	string	The name of the file with the
;				calibration image; at present this
;				must be a full path.
;
; Keywords:
;	store	??	input	If set, then add the calibration image
;				to the set of loaded images -- useful
;                               if you want to form differences
;                               from the calibration image.
;	all	??	input	If set, then normalize ALL images, by
;				default only the currently selected
;				images are normalized.
;	use_in_mem ?	input	If set, then use the specified image
;				from the currently loaded images.
;
; Restrictions:
;	The STORE key is ignored if USE_IN_MEM is specified.
;
; Side Effect:
;	Changes the values of the images.
;
; History:
;	Original: Mar 96; SJT
;-

@chandle.com
@wload.com

nimages = n_elements(h_array)
if (keyword_set(all)) then imlist = indgen(nimages) $
else imlist = sel_array
nimages = n_elements(imlist)

if (keyword_set(use_in_mem)) then begin
    dindex = (where(calib eq string_name, imatch))(0)
    if (imatch eq 0) then $
      message, "Specified image not found in loaded set." $
    else if (imatch gt 1) then  $
      message, /inform, "Multiple matches, using first."
    ghandle, dindex
    cal_im = image
    cal_head = head
endif else begin
    
    dindex = -1
    cal_im = readfits(calib, cal_head)
    
    if (keyword_set(store)) then begin
        head = cal_head
        image = cal_im
        inp = rstrpos(calib, '/')
        name = strmid(calib, inp+1, strlen(calib))
        chandle
        sel_img = [sel_img, 0]
    endif
endelse

cal_mask = (cal_im ne 0.0)
locs = where(finite(cal_im) ne 1, nnan)
if (nnan ne 0) then begin
    print, "Bad values in calibration image at:"
    print, locs
    stop
endif

;	Calibration images always contain the solar limb so that the
;	area close to the occulter can be excluded.

cal_cen = [sxpar(cal_head, 'CENTER_X'), $
           sxpar(cal_head, 'CENTER_Y')] ; This ought to be in
                                ; image pixels not shifted.


cal_tel = strtrim(sxpar(cal_head, 'DETECTOR'), 2)
case cal_tel of
    'C1': begin
        cff = 1.10
        dc = 322
    end
    'C2': begin
        cff = 2.00
        dc = 470
    end
    'C3': begin
        cff = 3.70
        dc = 319
    end
    
    Else: begin
        cff = 1.0               ; Shouldn't be needed unless we try
        dc = 0                  ; using this procedure for EIT
    end
endcase
cal_rad = float(sxpar(cal_head, 'RADIUS'))*cff ; OD radius

cal_sz = size(cal_im)
dx = findgen(cal_sz(1)) - cal_cen(0)
dy = findgen(1, cal_sz(2)) - cal_cen(1)
dr = sqrt(dx(*, intarr(cal_sz(2)))^2 + $
          dy(intarr(cal_sz(1)), *)^2) - cal_rad

cal_mask = cal_mask and (dr lt -5 or dr gt 15)


cal_bl = [sxpar(cal_head, 'P1COL'), sxpar(cal_head, 'P1ROW')]

print, '#', 'File', 'Exposure', 'Nominal', '% Diff', $
  format = "(3x,a1,2x,a16,4x,2a12,1x,a6)"

for j = 0, nimages - 1 do begin
    if (imlist(j) eq dindex) then goto, skip_image
    
    ghandle, imlist(j)
    
    bl = [sxpar(head, 'P1COL'), sxpar(head, 'P1ROW')]
    
    locs = where([sz(1:2), bl] ne [cal_sz(1:2), cal_bl], ndiff)
    
    if (ndiff ne 0) then begin
        print, "Image #", imlist(j)
        print, "Size or corner mismatch"
        print, "Calibration:", cal_sz(1:2), cal_bl
        print, "Image:      ", sz(1:2), bl
        print, "Skippping...."
        goto, skip_image
    endif
    
    pxscl = [sxpar(head, 'SUMCOL'), sxpar(head, 'SUMROW')] > 1

    mask = mask_out(image, pxscl, dark = dc)      ; Telemetry
    mask = rebin(temporary(mask), sz(1), sz(2), /samp)

    mask = mask and (image+dc) ne 16383 ; Saturation
    mask = mask and sigma_mask(image, 15, n_sigma = 4., $ ; Cosmic
                               /all, outbox = 3) ; rays & stars

    mask = mask and cal_mask
    
    used = where(mask, nvalid)
    
    exposure = total((mask(used) * image(used)) / cal_im(used)) / $
      float(nvalid)

    exp_nominal = float(chk_h4s(head, 'EXPTIME'))

    percent = (1 - exp_nominal/exposure)*100.
    print, imlist(j), name, exposure, exp_nominal, percent, $
      format = "(I4,2x,a20,2F12.5,f7.2)"
    
    if (abs(percent) gt 10.) then begin
Ctu:
        ans = ''
        read, "Nominal exposure is wrong by > 10% " + $
          "probable mismatch, continue (y/n/q) <N> :_", ans
        case strupcase(ans) of
            'N': goto, skip_image
            'Y':
            'Q': goto, abandon
            '': goto, skip_image
            Else: goto, ctu
        endcase
    endif

    image = image/exposure
    handle_value, h_array(imlist(j)), image, /set

Skip_image:

endfor

ld.net = 1

Abandon:

end

