
function hsi_pixon_fit_bgrnd,residuals,iobj,bobj,dobj,celist,a2d,harmonics, $
                             tty=ttyin,bwindow=bwindow,noadd=noadd, $
                             nooffset=nooffset,bflat=bflat

;+
;NAME:
;     HSI_PIXON_FIT_BGRND
;PURPOSE:
;     Fit sine waves to residuals to estimate background
;CATEGORY:
;CALLING SEQUENCE:
;     bobj = hsi_pixon_fit_bgrnd(residuals,iobj,bobj,dobj,celist,a2d,harmonic, $
;                                tty=tty,bwindow=bwindow)
;INPUTS:
;     residuals = residuals array
;     iobj = image object
;     dobj = data object (e.g. from hsi_pixon_get_data.pro)
;     bobj = background object (e.g. from hsi_pixon_get_data.pro)
;     celist = calibrated event list
;     a2d = list of detectors to use
;     harmonics = harmonics to to with the detector list
;OPTIONAL INPUT PARAMETERS:
;KEYWORD PARAMETERS
;     bflat = on putput has background in a flat array (rather than an
;             object).
;     / nooffset = force the minimum background to be 0.0
;     /tty = do plots
;     bwindow = window number to plot in it tty is set.
;     /noadd = do not add the new background estimate to the current
;              estimate givin in bobj.  By default they are added
;              together so that the program works iteratively.
;OUTPUTS:
;     bobj = new background object
;COMMON BLOCKS:
;SIDE EFFECTS:
;RESTRICTIONS:
;PROCEDURE:
;     Groups the residuals by roll angle and then fits sine waves with
;     pweriods equal to the spin period and twice the spin period to
;     the residual.
;MODIFICATION HISTORY:
;     T. Metcalf 2002-Nov-20
;     T. Metcalf 2003-May-07 Write backgroudn to celist.background pointer
;     K. Tolbert 2009-Jul-15. If a detector has no good data, abort with a
;       message instead of crashing.     
;-

   if n_elements(ttyin) LE 0 then tty=0 else tty = ttyin
   bflat = make_array(size=size(residuals),/float)

   if tty then begin
      wold = !d.window
      need_new_window = 0
      if n_elements(bwindow) LE 0 then need_new_window = 1 $
      else begin
         device, WINDOW = w
         if bwindow GT n_elements(w)-1 then need_new_window = 1 $
         else if NOT w[bwindow] then need_new_window = 1
      endelse
      if need_new_window then begin
         device,get_screen_size=scrsz
         window,/free,xsize=fix(640*(scrsz[1]/1200.)), $
                      ysize=fix(960*(scrsz[1]/1200.)), $
                      title='RHESSI Pixon Background Fits'
         bwindow = !d.window
      endif
      wset,bwindow
      psave = !p.multi
      !p.multi = [0,1,n_elements(a2d),1,1]
   endif
   time_bin = float(iobj->get(/time_bin_def)*iobj->get(/time_bin_min))/2L^20
   iresidptr = 0L
   for i=0L,n_elements(a2d)-1L do begin  ; Is this the right order???
      this_a2d = a2d[i]
      for j=0L,n_elements(*harmonics[i])-1L do begin
         this_harmonic = (*harmonics[i])[j]
         if ptr_valid(dobj[this_a2d,this_harmonic]) then begin
            flux_var = (*celist[this_a2d,this_harmonic]).flux_var
            nresiddat  = n_elements(*dobj[this_a2d,this_harmonic])
            stemp = residuals[iresidptr:iresidptr+nresiddat-1L]
            time = time_bin[this_a2d MOD 9]*findgen(n_elements(stemp))
            nodropout = where((*celist[this_a2d,this_harmonic]).livetime GT 0.001,nnodropout)
            if nnodropout eq 0 then message,'Detector ' + trim(this_a2d+1) + ' data are all zero.  Aborting.  Remove this detector from your selection.'
            ; Compute phase in the spacecraft rotation
            phase = (((*celist[this_a2d,this_harmonic]).roll_angle)[nodropout]) MOD (2.0*!pi)
            ; Bin the data by phase
            radpertbin = (((*celist[this_a2d,this_harmonic]).roll_angle)[n_elements(stemp)-1]-((*celist[this_a2d,this_harmonic]).roll_angle)[0])/(n_elements(stemp))
            binsize = ((8.*radpertbin) < (!pi/8.)) > radpertbin
            hphasehist = histogram(phase,binsize=binsize, $
                                   reverse_indices=ssphase)
            ; Now reverse the histogram to find the total
            ; residual at each phase of the roll
            hphase = make_array(size=size(hphasehist), $
                                /double,value=0.0d0)
            for ireverse=0L,n_elements(hphasehist)-1L do $
               if hphasehist[ireverse] NE 0 then $
                  hphase[ireverse] = $
                  total(stemp[nodropout[ssphase[ssphase[ireverse]:$
                                        ssphase[ireverse+1]-1]]], $
                        /double)/double(hphasehist[ireverse])
            ; Now fit the binned data to sine waves
            theta = (findgen(n_elements(hphase))+0.5)*binsize
            sigma = sqrt(abs(hphase))>1.
            ; treat zeroes as "no data" by raising sigma
            bad = where(hphase EQ 0.,nbad)
            if nbad GT 0 then sigma[bad] = (max(hphase)*1000.)>1.
            btemp1 = hsi_pixon_calc_bgrnd(theta,hphase,sigma, $
                                          2.*!pi,fit=fit)
            ; expand back to the original dimensions
            btemp = make_array(size=size(stemp),/float,value=0.0)
            btemp[nodropout] = fit[0]+fit[1]*sin(phase+fit[2]) + $
                                      fit[3]*sin(2.*phase+fit[4])
            if keyword_set(nooffset) then begin
               offset = fit[0]-abs(fit[1])-abs(fit[3])
               if tty then btemp1 = btemp1 - offset
               btemp[nodropout] = btemp[nodropout] - offset
            endif
                     
            if ptr_valid(bobj[this_a2d,this_harmonic]) then begin
               if NOT keyword_set(noadd) then $
                  btemp = btemp + (*bobj[this_a2d,this_harmonic])
               ptr_free,bobj[this_a2d,this_harmonic]
            endif

            ; The background should be restricted to positive values
            ; Is btemp>0. the best way to do that???
            ; Multiply by flux_var since the modulation profile is 
            ; multiplied by flux_var in the residuals calculation. 
            ; This makes the "units" of the background and the mod
            ; profile consistent.
            bobj[this_a2d,this_harmonic] = ptr_new((btemp*flux_var)>0.)
            bflat[iresidptr:iresidptr+nresiddat-1L] = *bobj[this_a2d,this_harmonic]
            ; Write the background to the celist to store it.
            (*celist[this_a2d,this_harmonic]).background = $
               *bobj[this_a2d,this_harmonic]
            iresidptr = iresidptr + nresiddat 
            if tty then begin
               hp_bt1 = hphase-btemp1
               plot,theta,hphase,charsize=2.0,xstyle=1, $
                    title=strcompress('Background Fit DET: '+ $
                                      string(a2d[i]+1)), $
                    yrange = [min([min(hp_bt1),min(hphase),min(btemp1)]), $
                              max([max(hp_bt1),max(hphase),max(btemp1)])]
               oplot,theta,btemp1,color=!d.table_size/2
               oplot,theta,hp_bt1,color=fix(3.*!d.table_size/4.)
               oplot,[0.0,2.0*!pi],[0.,0.],linestyle=1,color=fix(3.*!d.table_size/4.)
            endif
         endif
      endfor
   endfor
   if tty then begin
      !p.multi = psave
      wset,wold 
   endif

   return,bobj

end
