;+
; Name: hsi_image_profile
;
; Purpose: Generates profiles from images
;
; Input:
; obj - hsi_image, hsi_image_alg or hsi_modul_profile object.  Note:  When called internally while
;  making an image, for vis algs, input obj can't be the alg object (e.g. hsi_vis_fwdfit, which
;  inherits hsi_image_alg) because it doesn't have access to everything.  So for vis algs, pass
;  in its caller which is an hsi_image_single obj.  This is irrelevant when called directly by user,
;  who will pass in hsi_image obj.
;
; Input Keywords:
; vimage - image to get profile for in annsec coords.  If obj is an hsi_modul_profile object
;   then, vimage must be supplied
; ALL- if set, return profile for all 9 dets regardless of det_index_mask
; PTR - if set, return only the pointer array
; Output Keywords:
; xp - array of pointers to image profiles
; index - indices of the start and final positions for each pointer element
;		Similar to reverse_indices for histogram. Values of ptr[i] are in index[i]:index[i+1]-1 of
;		the returned profile, see ptr_concat.pro for more details
; valid - indices of the valid pointers, from the set of 0,1,3,...,8
; scaled_vimage - output. Scaled image used to get profiles
; Error - set if an image can't be found in obj and if vimage_in isn't passed
;		Function returns, -1 in this case
; Written: Ras, 25-jan-2011, extracted elements from hsi_image_profile_cstat
; Modifications:
;  18-jun-2017, RAS, added support for Cart maps by calling hsi_image_profile_all
;
;-

function hsi_image_profile, obj, vimage=vimage_in, xp=xp, index=index, valid=valid, $
  scaled_vimage=scaled_vimage,  all=all, error=error, ptr=ptr

  ; seed is saved in common so that subsequent runs keep using random numbers along sequence.
  ; Otherwise it repeats.  But if user passes in seed, will use that, so sequence is
  ; repeatable.
  error=1
  self = obj


  scaled_vimage = hsi_image_get_vimage(self, vimage=vimage_in, /mp_scale, /convert2annsec)
  if (n_elements(scaled_vimage) le 1) && (scaled_vimage[0] eq -1) then return,-1
  ;change from hsi_annsec_profile_all to hsi_image_profile_all
  if keyword_set(all) then xp = hsi_profile_all( scaled_vimage, obj) else $
    xp = self -> getdata(class='hsi_modul_profile', vimage=scaled_vimage)
  alg_struct = hsi_algorithm_units(obj, /used, status=status)
  if alg_struct.is_vis  && (self->get(/vis_normalize) eq 1) then begin
    for i=0,8 do begin
      corr = self->get(/ vis_corr_factors)
      if ptr_valid(xp[i]) then *xp[i] = f_div(*xp[i] , corr[i])
    endfor
  endif

  error = 0
  if keyword_set(ptr) then return, xp
  xpall = ptr_concat(xp, index, valid)
  if ~arg_present(xp) then ptr_free, xp
  return, xpall
end