;+
;
; Name: hsi_image_get_vimage
;
; Purpose: Return the image in annsec coordinates.
;
; Method:  If vimage isn't passed in, gets image from alg class.
;   If vimage is passed in, and alg is not vis, then just returns image.
;   If alg is vis, then the vimage passed in (or retrieved from image class)
;   is not in annsec coords, so we must first make sure that an hsi_modul_profile
;   class has been instantiated so that hsi_xy2annsec.  If it hasn't, the easiest
;   thing to do is to quickly make a bproj image (that will make an hsi_modul_profile
;   class) and then set alg back to one in use.  Then call hsi_xy2annsec to convert
;   image to annsec coords.
;   Didn't make this an object method because it works for several classes
;
; Input:
;   obj - hsi_image_single or hsi_image_alg (any of alg classes, e.g. clean, pixon) object
;
; Input keywords:
;   vimage - image from image alg class
;	CONVERT2ANNSEC - image is in hsi_image->getdata() units, ie, cartesian coord in photon/cm2/asec^2
;		and must be converted to annsec coordinates and counts/subcoll
;	MP_SCALE - If set, then return in the units for use with hsi_modul_profile::getdata(vimage=vimage)
;		set to units of pixon->getdata(), 'Counts s!u-1!n sc!u-1!n'
; Output: Image in annsec coords (-1 if can't do it)
;
; Written: Kim Tolbert April 2010
;	Modified
;		26-jan-2011, ras, added CONVERT2ANNSEC to support converting photon images from
;		hsi_image when the image_alg class image is not available.
;	6-jun-2017, ras, convert2annsec forced to zero for cartesian modul pattern strategy
;
;-

function hsi_image_get_vimage, obj, vimage=vimage_in, convert2annsec = convert2annsec, mp_scale=mp_scale

self = obj
is_cart = stregex( /bool, /fold, hsi_get_modpat_strategy( obj ), 'cart')
if is_cart then convert2annsec = 0
alg_struct = hsi_algorithm_units(obj, /used, status=status)
if ~status then return, -1

if exist(vimage_in) then vimage = vimage_in else begin
  ;if obj is hsi_image_alg then it's already of class 'class', so don't need class keyword set
  if ~is_class(obj, 'hsi_image_alg') then class = alg_struct.class
  vimage = self -> getdata(class=class)
  convert2annsec = 0 ;unless alg_struct.is_vis then already annsec
endelse

if alg_struct.is_vis or keyword_set(convert2annsec) then begin
  if is_class(self, 'hsi_image_alg') then self = obj->getcaller()
  ; now self is hsi_image_single.  If modul_profile class isn't available
  ; make it by making a bproj image
  if ~obj_valid(self -> get(/obj,class='hsi_modul_profile')) then begin
    dum = self -> getdata(image_alg = 'bproj')
    self -> set, image_alg=alg_struct.name
  endif

  ; convert vimage to annsec coords
  vimage_annsec = hsi_xy2annsec(vimage,self)
  if ~same_data(size(vimage_annsec), size(vimage)) then begin ;if same size we didn't need to convert
	  vimage = vimage_annsec
	  if keyword_set(mp_scale) then begin
	  	vimo= vimage * hsi_alg_scale(self, alg_struct.class, status=status)

	  	return, vimo
	  	endif
  endif
endif

;If MP_SCALE is set, then return in the units for use with hsi_modul_profile::getdata(vimage=vimage)
if keyword_set(mp_scale) then return, vimage * hsi_alg_scale(self, status=status)


return, vimage

end