;+
; PROJECT:
;	HESSI
; NAME:
;	HSI_CART_MODUL_PATTERN
;
; PURPOSE:
;	This procedure calculates various cartesian image response building blocks.
;
; CATEGORY:
;	HESSI, IMAGE
;
; CALLING SEQUENCE:
; hsi_cart_modul_pattern,  obj, cstrx, cmpat, $
; weight = weight, _extra = _extra
; CALLS:
;	HESSI OBJECTS
;
; INPUTS:
;      obj - object containing calibrated eventlist object.
;
;
;
; OUTPUTS:
; cstrx : concatenated calib eventlist structure with additionally precomputed tags for bproj and profile
;      { hsi_calib_event_stack_plus00, $
;        inherits hsi_calib_event_stack, $
;        time_bin: 0.0, $ time width in seconds, one value for each sub-coll, repeated
;        vrate0: 0.0, $ ;livetime_bin(sec) * gridtran * fluxvar * det_eff.rel
;        cvrate: 0.0, $ ; vrate0 * cos(phase_map_ctr) * modamp * (1-gap)
;        svrate: 0.0, $ ; vrate0 * sin(phase_map_ctr) * modamp * (1-gap)
;        macosphz: 0.0, $ ; cos(phase_map_ctr) * modamp
;        masinphz: 0.0 $ ; sin(phase_map_ctr) * modamp
;      }
;    IDL> help, cstrx,/st
;    ** Structure HSI_CALIB_EVENT_STACK_PLUS00, 16 tags, length=60, data length=59:
;    ROLL_ANGLE      FLOAT         0.0490874
;    MODAMP          FLOAT           1.11610
;    PHASE_MAP_CTR   FLOAT          0.258312
;    GRIDTRAN        FLOAT          0.144507
;    FLUX_VAR        FLOAT          0.895600
;    BACKGROUND      FLOAT          0.000000
;    COUNT           FLOAT           3.00000
;    LIVETIME        FLOAT           50.1348
;    GAP             BYTE         0
;    ATTEN_STATE     INT              1
;    TIME_BIN        FLOAT       0.000976563
;    VRATE0          FLOAT        0.00617640
;    CVRATE          FLOAT        0.00666481
;    SVRATE          FLOAT        0.00176094
;    MACOSPHZ        FLOAT           1.07908
;    MASINPHZ        FLOAT          0.285108;	cmpat - annular sector map structure. partially complete.
;    IDL> help, cmpat
;    ** Structure HSI_CART_MODUL_PATTERN_STR, 18 tags, length=132, data length=131:
;    COS_MAP0        POINTER   <PtrHeapVar48973> - used with profile function
;    COS_MAP1        POINTER   <PtrHeapVar48974> - transpose of cos_map0, used for bproj
;    SIN_MAP0        POINTER   <PtrHeapVar48975>
;    SIN_MAP1        POINTER   <PtrHeapVar48976>
;    MACOSPHZ        POINTER   <PtrHeapVar48977>
;    MASINPHZ        POINTER   <PtrHeapVar48978>
;    IMAGE_DIM       INT       Array[2]
;    PIXEL_SIZE      FLOAT     Array[2]
;    PIXEL_SCALE     FLOAT           1.00000
;    NMAPS           LONG                 6
;    NMAP            LONG      Array[7]
;    NPHZ            LONG                12
;    DET_INDEX_MASK  BYTE      Array[9]
;    HARMONIC        INT              1
;    DET_EFF_REL     FLOAT     Array[6]
;    DET_EFF_AVG     FLOAT          0.548151
;    ENERGY_BAND     FLOAT     Array[2]
;    XYOFFSET        FLOAT     Array[2];
;
; OPTIONAL OUTPUTS:
;	none
;
; KEYWORDS:
;	 weight - detector weights
;
; COMMON BLOCKS:
;	none
;
; SIDE EFFECTS:
;	none
;
; RESTRICTIONS:
;	none
;
; PROCEDURE:
;	Builds cosine and sine modulation pattern maps for phase stacked calibrated eventlists for the rotation angle and
;	phase of the first element in a stack group (same rot angle for 12 phases)
;
; MODIFICATION HISTORY:
;	20-aug-2015, richard.schwartz@nasa.gov

;-
pro hsi_cart_modul_pattern,  obj, cstrx, cmpat, $
  weight = weight, all=all, this_det_index = this_det_index, _extra = _extra

  if keyword_set( _extra ) then obj->set, _extra = _extra
  default, this_det_index, -1
  test = is_object( obj )
  if test then begin
    cntrl = obj->get(/xyoffset, /image_dim, /pixel_size, /pixel_scale, $
      /det_index_mask, /use_phz_stacker, /phz_n_phase_bins, /cbe_det_eff, /energy_band)
    confirm_these = ['image_dim','xyoffset','use_phz_stacker','energy_band','cbe_det_eff']
    valid = have_tag( cntrl, confirm_these, count = count)
    valid and= count eq n_elements( confirm_these )
  endif
  if ~test[0] || ~valid then message,'Must be an calib_eventlist object with phase stacking'

  pixel_size = cntrl.pixel_size
  pixel_scale = cntrl.pixel_scale
  det_index_mask = cntrl.det_index_mask
  nphz = cntrl.phz_n_phase_bins
  det_eff = cntrl.cbe_det_eff
  energy_band = cntrl.energy_band
  image_dim = cntrl.image_dim

  default, weight, fltarr(9) + 1.0

  cobj =  obj->get(/obj,class='hsi_calib_eventlist')

  if not exist(image_dim) or not exist(pixel_size) or not exist(pixel_scale) then $
    message,'Need image obj or modul pattern controls:  image_dim, pixel_size, pixel_scale'
  pixel_size = pixel_size * pixel_scale[0]
  if keyword_set( all ) or this_det_index ge 0 then begin
    cbeptr = cobj->framework::getdata()
    cbeptr = this_det_index lt 0 ? cbeptr : cbeptr[ this_det_index ]
    det_index_mask =  bytarr(9) + 1
    if this_det_index lt 0 then begin
      det_index_mask *=0
      det_index_mask[ this_det_index ]++
    endif
  endif else cbeptr = cobj->getdata()
  calib_str = ptr_concat(cbeptr, nbin, sel, nsel)
  ;if in single this_det_index_mode change sel to this_det_index
  sel = this_det_index ge 0 ? this_det_index : sel

  npix = product( image_dim )
  nbins = nbin[ nsel ]
  nmap  = nbin/nphz ;start index of valid maps (0-!pi only, cos and sin)
  nmaps = nbins/nphz ;total valid maps (1/2 from 0-!pi and 2 from cos,sin)
  det_list = this_det_index ge 0 ? [ this_det_index ] : where( det_index_mask )

  for i = 0, nsel-1 do det_id = append_arr( det_id, bytarr( nmap[i+1]/2-nmap[i]/2 ) + det_list[i])
  time_bins = hsi_get_time_bins( cobj, /sec)
  time_bins = float( time_bins[ sel ] )
  det_eff = cobj->get(/cbe_det_eff)
  det_eff_rel = det_eff.rel[sel]
  ;Add new fields to calib_str for faster computation

  cstrx = replicate( { hsi_calib_event_stack_plus00, $
    inherits hsi_calib_event_stack, $
    time_bin: 0.0, $ time width in seconds, one value for each sub-coll, repeated
    vrate0: 0.0, $ ;livetime_bin(sec) * gridtran * fluxvar * det_eff.rel
    cvrate: 0.0, $ ; vrate0 * cos(phase_map_ctr) * modamp * (1-gap)
    svrate: 0.0, $ ; vrate0 * sin(phase_map_ctr) * modamp * (1-gap)
    macosphz: 0.0, $ ; cos(phase_map_ctr) * modamp
    masinphz: 0.0 $ ; sin(phase_map_ctr) * modamp
  }, nbins )

  struct_assign, calib_str, cstrx
  for i = 0, nsel -1 do begin
    temp = cstrx[ nbin[i] : nbin[i+1]-1 ]
    temp.time_bin = time_bins[i]
    temp.vrate0 = temp.livetime * temp.gridtran * temp.flux_var * det_eff_rel[i] * time_bins[i]
    temp.macosphz = cos( temp.phase_map_ctr ) * temp.modamp * (1 - temp.gap) * weight[ sel[i] ] ;used in bproj
    temp.masinphz = sin( temp.phase_map_ctr ) * temp.modamp * (1 - temp.gap) * weight[ sel[i] ]
    temp.cvrate = temp.vrate0 * temp.macosphz
    temp.svrate = temp.vrate0 * temp.masinphz
    cstrx[ nbin[i] : nbin[i+1]-1 ] =temp
  endfor

  cos_sin   = fltarr( npix, nmaps )
  ;compute the cos and sin maps of the phase maps for each pixel in the fov, cos_sin varies from -1 to 1
  for i=0,nsel-1 do $
    cos_sin[ 0, nmap[i] ]  = $
    hsi_cart_mpati_stk( cbeptr, sel[i], nphz, image_dim, pixel_size * pixel_scale, pixel_coord) ;reuse pixel_coord
  ;Now concatenate all the cos maps across sub-colls and all the sin maps.
  ;These are only from 0 to !pi
  dmap = (nmap[1:*] - nmap ) /2
  all_cos = fltarr( npix, nmaps/2)
  all_sin = all_cos
  for i = 0, nsel-1 do all_cos[ 0, nmap[i] / 2] = cos_sin[*, nmap[i] : nmap[i] + dmap[i] - 1 ]
  for i = 0, nsel-1 do all_sin[ 0, nmap[i] / 2] = cos_sin[*, nmap[i] + dmap[i] : nmap[i+1] - 1 ]
  ;tranpose all_cos (cos_map0 and sin_map0) and macosphz and masinphz are for fast compute of bproj

  cmpat = {$
    det_id: ptr_new(det_id), $
    cos_map0: ptr_new(  all_cos  ),  $
    cos_map1: ptr_new( transpose( all_cos )), $
    sin_map0: ptr_new(  all_sin  ),  $
    sin_map1: ptr_new( transpose( all_sin )), $
    vrate0: ptr_new( cstrx.vrate0), $
    cvrate: ptr_new( cstrx.cvrate), $
    svrate: ptr_new( cstrx.svrate), $
    weight_map_ptr: ptr_new(), $ ;to be filled in later by
    ; hsi_cart_pattern_weight_map_build 

    macosphz: ptr_New( cstrx.macosphz ), $
    masinphz: ptr_new( cstrx.masinphz ), $
    image_dim: image_dim, $
    pixel_size: pixel_size, $
    pixel_scale: pixel_scale, $
    nmaps: nsel, $
    nmap: nmap, $
    nphz:nphz, $
    this_det_index: this_det_index, $ ;if this is -1 then more then it's aggregated
    det_index_mask: det_index_mask, $
    det_eff_rel: det_eff_rel, $
    det_eff_avg: det_eff.avg, $
    energy_band: energy_band, $
    xyoffset: obj->get(/xyoffset) $   ;property of pointing, not modulation patterns
  }
end
