;+
; PROJECT:
;   HESSI
; NAME:
;   hsi_cart_bproj_arr
;
; PURPOSE:
;   Returns the back-projection of the hessi calibrated eventlist.
;
; CATEGORY:
;   HESSI, UTIL
;
; CALLING SEQUENCE:
;   map = hsi_cart_bproj_arr( obj_in, map_ptr = fifo, /sum, det_index_mask = det_index_mask )
;
;   Call it this way when the modulation pattern maps are carried outside of the object.
;   This mode is used for testing.
;
;   map = hsi_cart_bproj_arr( obj_in, map_ptr = map_ptr,/sum )
;
;
; CALLS:
;   none
;
; INPUTS:
;       obj_in - Object containing the calibrated eventlist or a pointer array to same.  If computing this for
;     a set of detectors without the THIS_DET_INDEX  keywords, then either
;     the cbe containing object or the cbe pointer + DET_INDEX_MASK + rmap_dim must be defined.
;
; OPTIONAL INPUTS:
;
;
; OUTPUTS:
;       Returns the map as a pointer ( if SUM eq 0 ) or as a 2-d image in polar coordinates.
;
; OPTIONAL OUTPUTS:
;   none
;
; KEYWORDS:
;   MAP_PTR - keyword only to be used in testing.  Normally modulation pattern maps are carried
;   in the hsi_modul_pattern object, but for testing purposes ( see hsi_annsec_test.pro ) they
;   are passed around through this keyword.
;   SUM - If set, sum the map over the detectors.
;   THIS_DET_INDEX- detector index for single collimator/harmonic pair.
;   20-apr-2007 ras, removed ;THIS_HARMONIC- harmonic index for single collimator/harmonic pair.
;   DET_INDEX_MASK - Specify array of collimators and harmonics. See hsi_calib_eventlist.
;   DATA_PTR - Alternate count rate input.
;     E.G. Used to create point spread functions and intermediate MEM images.
;     Counts per time_bin corrected for live_time. May be a vector if not for
;     an array of detectors.
;   COUNTS_SUMMED - sum of livetime corrected counts. Returned as a 9 elem array
;   unles this_det_index is used, when it comes back as a single number.
;   USE_RATE- use count rate not count to for back-projection weighting.
;	MAX_HARMONIC - normally harmonics are detected from modulation pattern structure
;		but for flatfield correction need to force use only 1st harmonic for simplicity
;
;
; COMMON BLOCKS:
;   none
;
; SIDE EFFECTS:
;   none
;
; RESTRICTIONS:
;   none
;
; PROCEDURE:
;   none
;
; MODIFICATION HISTORY:
;   Version 1, richard.schwartz@gsfc.nasa.gov
;   30-mar-2000
;   Version 2, richard schwartz, det_index_mask and a2d_index_mask are anded. 2-may-2000.
;   Version 3, ras, uses new direct indexing ( see hsi_annsec_map ) and phase_ptr.isum field.
;   Version 3.1, ras, 25-may-2000
;   - Added protection against no data in calib_eventlist for this_det_index & this_harmonic.
;   Version 5, richard.schwartz@gsfc.nasa.gov 17-jul-2000.
;   - switching to interpolating on base maps. A modulation pattern is
;   composed of the weighted sum of four base maps. The phases are adjusted in the two sin/cos pairs to
;   match the phase of the real modulation pattern.  The weighting is assigned to the separate pairs based on
;   the distance of each ( in rotation angle ) from the rotation angle at the center of the event bin.
;   Version 5.1, richard.schwartz@gsfc.nasa.gov 3-aug-2000. Return flat map of 0 and 0 counts_summed
;   for 0 counts in data_ptr or cbe input.
;   Version 5.2, richard.schwartz@gsfc.nasa.gov, 18-aug-2000. Optimized the order of the radius-row/time-row
;   matrix.
;   Version 5.3, richard.schwartz@gsfc.nasa.gov - new formulation by GH dropping gridtran term
;   from inside the averages.- 19-dec-2000.
;   Version 5.4, richard.schwartz@gsfc.nasa.gov, 20-dec-2000, eliminate needless a2d_index checks.
;      11-jun-2001, richard.schwartz@gsfc.nasa.gov, propagate det_eff    ;
;   Aug 25, 2001, richard.schwartz@gsfc.nasa.gov - remove alternative computations of back projection.
;     These methods relied on the first index of the cmap and smap coinciding with the first
;     time( phase ) interval.  For the universal patterns, this synch is broken.
;   1-apr-02, kat, modified null return
;   24-apr-02, ras, sped up association between data time bins and used modpats.
;   12-jan-2004, ras, get cbe_ptr before map_ptr instead of reverse
;   18-apr-2004, ras    bin culling included and use_rate keyword added, smoothing_angle added
;     smoothing_angle is for hsi_annsec_bproj_weight_map, to reduce systematics in numerator
;     term of flatfielding.  This should help compute the pedestal that is subtracted from the
;     back-projection leaving only the modulating fraction
;   22-feb-2005, ras, fixed action of culling, restricted to zeroing vrate and not removing
;     base maps from the first 180 degrees
;    12-may-2005, ras, added DET_INDEX_OFF check for times
;     with no livetime for a particular detector such as 8
;     Won't be set on the first pass through hsi_bproj_annsec so must check inside
;     hsi_annsec_bproj.pro as well.
;    28-jun-2005, ras, changed DET_INDEX_OFF to OFF_DET_INDEX
;	 20-apr-2007, ras, removed this_harmonic and compute with harmonics if present
;		added max_harmonic - need to suppress higher harmonics when computing flatfield
;		correction because map^2 term to complex for efficient computation with higher
;		harmonics
;		20-jul-2007, RAS, Required by annsec formulation - Self->set, EMPTY_FLAG = 0
;	 29-Nov-2007, Kim. ocbe->get( /need_update ) doesn't work. Changed to ocbe->need_update(  )
;	17-feb-2010, RAS, EMPTY_FLAG no longer needed, when livetime field is 0, so is count
;	2-jan-2015, ras, added cbe_ptr to speed repeated calls
; 28-mar-2017, ras,  tidied up
; 4-apr-2017, ras, added spatial_frequency_weight and code to handle using it to make the
; weighted sum of sub-coll bproj into a single image
;-
function hsi_cart_bproj_arr,  cbe_ptr,  $
  cmap_ptr, data_ptr=data_ptr, $
  use_rate = use_rate, $
  counts_summed = counts_summed, $
  flatfield = flatfield
  
  valid = where( ptr_valid( cbe_ptr ) * ptr_valid(cmap_ptr), nvalid )
  counts_summed = 0.0
  default, data_ptr, ''
  for i = 0, nvalid -1 do begin
    ii = valid[i]
    data = ptr_chk( data_ptr ) ? *data_ptr[ii] : (is_number( data_ptr[0]) ? data_ptr :"")
    cmap = *cmap_ptr[ ii ]
    out  = exist( out ) ? out : fltarr( [ cmap.image_dim, nvalid ] )
    out[0,0,i] = hsi_cart_bproj_1det( *cbe_ptr[ii], cmap, data, $
      use_rate = use_rate, no_tot_vrate = no_tot_vrate, counts_summed = summed_counts_ii, $
      flatfield = flatfield )
    counts_summed += summed_counts_ii
  endfor
  ;don't return pointer array, return float( nx, ny, ndet)
  return, out

end


