;+
; Name: vis_multiscale_psf_gen
;
; Purpose: This function computes the cross-convolutions between basis-functions and PSFs at 
;          different scales (RHESSI)
;
; Inputs:
;   - vis - visibility bag
;   - nscale - number of different scale-function that compose the map
;   - pixon - basis functions
;
; Keyword inputs:
;   - npixel       number of pixels in x and y, 1 or 2 element long vector or scalar
;                  images are square so the second number isn't used
;   - pixel_size   pixel size in asec (pixels are square)
;   - detect       a 9-element vector containing for each detector the corresponding scale
;                  (i.e [1 1 2 2 3 3 3 4 4] means detectors 1 and 2 at scale 1; detectors 3 and 4 at scale 2;
;                  detectors 5, 6 and 7 at scale 3; detectors 8 and 9 at scale 4)
;   - init_psf     1b to compute the psf 
;                  0b to use the pre-computed psf 
;   
; Keyword outputs:
;   - psf             psf associated with each scale
;   - max_dirty_pixon vector containing the maximum values of the convolutions of basis functions and PSFs
;                     at the same scale (i.e., max_dirty_pixon[i]=max(convolve(psf[*,*,i], pixon[*,*,i]))

function hsi_vis_multiscale_psf_gen, vis, nscale, pixon, npixel = npixel, pixel_size = pixel_size, psf = psf, $
                                 max_dirty_pixon = max_dirty_pixon, init_psf = init_psf, detect = detect

  default, init_psf, 1
  dim_psf = size(/dim, pixon[*,*,0])/2 * 2 +1
  default, npixel, dim_psf[0]
  default, pixel_size, 1.0
  default, detect, [1,1,1,2,2,2,3,3,3]
  
  basis = replicate( { map:fltarr( npixel, npixel) }, nscale)
  space = replicate( {basis: basis}, nscale )
  max_dirty_pixon = fltarr( nscale )
  nvis = N_elements( vis )
  
  if init_psf then begin
  
    psf = fltarr( npixel, npixel, nscale )
  
    for ibasis = 0, nscale - 1  do begin
 
      ind = where( detect eq ibasis+1 )
      visin = vis[where( vis.isc GE min( ind ) and vis.isc LE max( ind ), countvis )]
  
      dummy = vis_psf( visin, image_dim = [npixel, npixel], pixel = pixel_size, /init)
      psf[*,*,ibasis] = dummy*countvis/nvis
      
      for jspace = 0, nscale - 1 do begin
        space[jspace].basis[ibasis].map = convolve( psf[*,*,ibasis], pixon[*,*,jspace] )
      endfor

      max_dirty_pixon[ibasis] = max( space[ibasis].basis[ibasis].map )
    
    endfor
    
  endif else begin
    
    for ibasis = 0, nscale-1 do begin  
    
      for jspace = 0, nscale - 1 do begin
        space[jspace].basis[ibasis].map = convolve( psf[*,*, ibasis], pixon[*,*,jspace] )
      endfor
      max_dirty_pixon[ibasis] = max( space[ibasis].basis[ibasis].map )

    endfor
  
  endelse
  
  return, space
end
