;+
; Name: hsi_vis_multiscale_pixon_gen
;
; Purpose: This function computes the basis functions using visibilities (RHESSI)
;
; Inputs:
;   - vis - visibility bag
;
; Keyword inputs:
;   - npixel       number of pixels (scalar);                 
;   - pixel_size   pixel size in arcsec (pixels are square)
;   - detect       a 9-element vector containing for each detector the corresponding scale
;                  (i.e [1 1 2 2 3 3 3 4 4] means detectors 1 and 2 at scale 1; detectors 3 and 4 at scale 2;
;                  detectors 5, 6 and 7 at scale 3; detectors 8 and 9 at scale 4)
;   - gauss_basis  1b for gaussian basis-function
;                  0b for parabolic basis-function
;
; Keyword outputs:
;   - psf          psf associated with each scale
;   - dim_pixon    dimension of the basis function

function hsi_vis_multiscale_pixon_gen, vis, npixel = npixel, pixel_size = pixel_size, $
                                   psf = psf, dim_pixon = dim_pixon, detect = detect, $
                                   gauss_basis = gauss_basis, quiet = quiet

  default, npixel, 65
  default, quiet, 0
  default, pixel_size, 1.0
  default, nscale, 3
  default, detect, [1,1,1,2,2,2,3,3,3]
  default, gauss_basis, 0b
  
  nscale = max( detect )
  scale = fltarr( nscale )
  psf = fltarr( npixel, npixel, nscale )
  
  nvis = N_elements( vis )
  
;  !p.multi = [0,1,nscale]
;  window, 1, xsize = 600, ysize = 700
  
  for ibasis = 0, nscale - 1  do begin

    ind = where( detect eq ibasis+1 )
    visin = vis[where( vis.isc GE min( ind ) and vis.isc LE max( ind ), countvis )]

    psf[*,*,ibasis] = vis_psf( visin, image_dim = [npixel, npixel], pixel = pixel_size, /init ) * countvis / nvis
    max_psf = max( psf[*,*,ibasis], xy)
    ind = get_ij( xy, npixel )
    psf_cut = psf[*, ind[1], ibasis]
    
    if min( psf_cut ) LT 0 then begin
      bound_left = max( where( psf_cut[0:ind[1]] LT 0 ) )
      bound_right = ind[1]+min( where( psf_cut[ind[1]:npixel-1] LT 0 ) )
    endif else begin
      bound_left = 0
      bound_right = npixel-1
    endelse
      
    dummy0 = min( abs( psf_cut[bound_left:ind[1]]-max_psf/2. ),ind1 )
    dummy0 = min( abs( psf_cut[ind[1]:bound_right]-max_psf/2. ),ind2 )
    ind1 = ind1+bound_left
    ind2 = ind2+ind[1]
   
    ;;;; fwhm in pixel unit of psf at scale ibasis
    if ind2-ind1 eq 0 then scale[ibasis]=1 else scale[ibasis] = ind2-ind1  

;    plot, psf_cut, xrange = [0,npixel-1], xstyle = 1, yrange = [min( psf[*,*,ibasis] ), max_psf], ystyle = 1
;    oplot, [ind1, ind2], [max_psf/2.,max_psf/2.]
  endfor
  
; !p.multi=0


  if gauss_basis then begin
    sigma = max( scale ) / ( 2. * sqrt( 2. * alog( 2. ) ) )
    dim_pixon = floor( 3. * sigma * pixel_size ) / 2 * 2 +1 
    if ~quiet then print,sigma, dim_pixon
    if dim_pixon LT npixel then dim_pixon = npixel
    if dim_pixon GT 255 then dim_pixon = 255
    if ~quiet then print,'(Gaussian) Pixon dimension: ', dim_pixon
    pixon = fltarr( dim_pixon, dim_pixon, nscale )
    for i = 0, nscale-1 do begin
      dummy = psf_gaussian( npixel = dim_pixon, FWHM = scale[i], ndim = 2 )
      pixon[*,*,i] = dummy / total(dummy)
    endfor
  endif else begin
    ;;;;;; radius (at z=0) in pixel units for a paraboloid vith max_psf as vertex and scale as fwhm
    radius = scale / sqrt( 2. )
    pixon = pxn_psf_arr( radius )
    for i = 0, nscale-1 do begin 
      pixon[*,*,i] /= total( pixon[*,*,i] )
    endfor
    dim_pixon = size( /dim, pixon ) 
    dim_pixon = dim_pixon[0]
    if ~quiet then print,'(Paraboloid) Pixon dimension: ', dim_pixon
  endelse
 
;  if gauss_basis then title_string = 'Gaussians' else title_string = 'Paraboloids' 
;  !p.multi = [0,nscale,1]
;  window, 2, xsize = 1000, ysize = 400, title = 'Basis functions: '+title_string
;  for i = 0, nscale-1 do plot_image, pixon[*,*,i]
;  !p.multi = 0
  
  return, pixon
end
